#!/bin/bash

# This is Rick Jerz's backup script for his production Moodle on his VPS.
# It keeps x days of backup files, defined by num_versions.
#   Simple-minded approach... use the date to delete old files.
# Last revised: 2025-21-01. (Revised for Moodle51)

# Uses today's date as a prefix.

# Future improvements:
# 	Record an entry into a 'log' file.
# 	Use the file "count" instead of the date.
#		Purge caches before backing up moodledata.

# #################################################
# When sharing, the user must provide the moodle_vars.txt file.
# Edit the variables below to match your Moodle environment.
# Modify the variables in the moodle_vars.txt file for your database.
# #################################################

# Variables
# --------------------------------------------------
# Note that, since this script is standalone, some variables from the upgrade script are repeated.

# Today's date
NOW=$(date +%F)
# Timer
((start = SECONDS))
#  Moodle version number (4, 5, 51, blank "", etc.)
mdl_version=5
# Number of backups.
num_versions=7
# Location of the Moodle code (i.e, the mooodle app)
mdl_code_folder=moodle$mdl_version
# Moodle code location
mdl_code_location=~/mdl_sites # VPS
# Location of "moodledata"
moodledata=~/mdl_data/moodledata$mdl_version # VPS
# Location of backup script and vars.txt file.
vars_folder=~/scripts/ # VPS
# The vars_file is the full path to the variables (i.e. database credentials) file.
vars_file=$vars_folder"m"$mdl_version"_vars.txt"
# Location for storing the backups. Create this folder before running this script.
mybu_dir=~/backups/m$mdl_version # VPS
# I back up the database, moodledata folder, moodle app folder, and config.php.
# These variables determine the names of the backup files.
mybu_dbname="moodledb$mdl_version.sql"
mybu_dataname="moodledata$mdl_version"
mybu_mdappname="moodle$mdl_version"
mybu_config="config$mdl_version"

# ------------ Begin the Backup ---------------------------
echo
echo "-------------------------------"
echo "Backup Moodle$mdl_version files."
echo "-------------------------------"

# Test to verify that MySQL is running; if not, exit.
#		Needed in MAMP, but my VPS server the database is always running.
#UP=$(pgrep mysql | wc -l)
#if [ "$UP" -ne 1 ]; then
#	echo "MySQL is down."
#	exit
#fi

# Read the "credentials" variables. If it doesn't exist, stop.
if [ ! -f "$vars_file" ]; then
	echo "Missing credentials file."
	exit
fi
echo "Using credentials from "$vars_file

#Load the variables from the text file.
source "$vars_file"

echo "Current backup date = "$NOW
# x days ago (change x to the number of backups)
WEEKAGO=$(date --date=$num_versions" day ago" +%F) # VPS
#WEEKAGO=$(date -v -7d +%F) # MacOS
echo "Oldest backup date = "$WEEKAGO

echo
echo "Backups are located in:"
echo "   "$mybu_dir

# Backup up 3 components of Moodle.
echo "1. Backup the Moodle$mdl_version database."
mysqldump -u$mdl_dbuser -p$mdl_dbpass --opt --no-tablespaces -r$mybu_dir/$NOW-$mybu_dbname $mdl_dbname

echo "2. Backup the Moodle$mdl_version moodledata$mdl_version folder."
# Note, you don't need all files.
# Purge caches (optional)
# tar cfz mytar.gz moodledata/filedir moodledata/geoip moodledata/lang, for example
tar cfz $mybu_dir"/$NOW-$mybu_dataname.gz" -P $moodledata

echo "3. Backup the Moodle$mdl_version code folder, $mdl_code_folder."
tar cfz $mybu_dir"/$NOW-$mdl_code_folder.gz" -P $mdl_code_location/$mdl_code_folder/

echo "4. Backup the Moodle$mdl_version config.php file."
cp $mdl_code_location/$mdl_code_folder/config.php $mybu_dir/$NOW-$mybu_config.php

# Delete the old (x days) files first
rm -f $mybu_dir/$WEEKAGO-$mybu_config.php
rm -f $mybu_dir/$WEEKAGO-$mybu_dbname
rm -f $mybu_dir/$WEEKAGO-$mybu_mdappname.gz
rm -f $mybu_dir/$WEEKAGO-$mybu_dataname.gz

# Show files in the backup directory that contain "202," (for year).
echo
echo "Here are all the backups in $mybu_dir."
ls -alh "$mybu_dir" | grep 202
echo

# Finished
((end = SECONDS)) # Used for overall timing
((duration_sec = end - start))
((duration_min = 1 + duration_sec / 60))
echo "*****************   Backup complete!   ***************** "
# Eventually, write this message to the log file.
echo $(date)": $mdl_code_folder backed up."
echo "Total backup duration = $duration_sec seconds. Under $duration_min minutes elapsed time for this process."
echo
